# Multiple stores

# celery -A scraper_worker.celery_app worker -Q campus_queue --concurrency=2 --prefetch-multiplier=1 --loglevel=info

# celery -A scraper_worker.celery_app worker -Q trends_queue --concurrency=2 --prefetch-multiplier=1 --loglevel=info

from celery import Celery
import requests
import os
import json
import time

# FastAPI URL that calls Selenium driver
API_URL = "http://127.0.0.1:8000/scrape-by-url"

# Directory to save output files
OUTPUT_DIR = "scraped_data"
os.makedirs(OUTPUT_DIR, exist_ok=True)

# Celery app setup with Redis broker and backend
celery_app = Celery(
    "scraper_tasks",
    broker="redis://localhost:6379/0",
    backend="redis://localhost:6379/0",
)

@celery_app.task(bind=True)
def scrape_task(self, params: dict, row_id: int, brand: str):
    """
    Celery task to hit /scrape-by-url endpoint (Selenium-based),
    get place data + bandwidth, and store in file per row.
    """
    try:
        print(f"[{brand.upper()}] 🚀 Row {row_id} - Starting scrape...")

        response = requests.get(API_URL, params=params, timeout=150)

        if response.status_code == 200:
            result = {
                "row_id": row_id,
                "params": params,
                "data": response.json()
            }

            file_name = f"{brand.lower()}_result_{row_id}.json"
            file_path = os.path.join(OUTPUT_DIR, file_name)

            with open(file_path, "w", encoding="utf-8") as f:
                json.dump(result, f, indent=2)

            print(f"[{brand.upper()}] ✅ Row {row_id} scraped successfully.")
            print(f"[{brand.upper()}] 📶 Bandwidth: {result['data'].get('bandwidth_used_mb', '?')} MB")

        else:
            print(f"[{brand.upper()}] ❌ Row {row_id} failed: {response.status_code} - {response.text}")

    except Exception as e:
        print(f"[{brand.upper()}] ❌ Row {row_id} exception: {str(e)}")
