import json
import asyncio # Changed from time
import re
from playwright.async_api import async_playwright, TimeoutError as PlaywrightTimeoutError # Changed to async
from urllib.parse import urlencode
from math import radians, sin, cos, sqrt, atan2

# Import the extraction functions from our helper module
from . import extractor

# --- Constants ---
BASE_URL = "https://www.google.com/maps/search/"
DEFAULT_TIMEOUT = 30000  # 30 seconds for navigation and selectors
SCROLL_PAUSE_TIME = 1.5  # Pause between scrolls
MAX_SCROLL_ATTEMPTS_WITHOUT_NEW_LINKS = 5 # Stop scrolling if no new links found after this many scrolls

## Calculate distance 

def haversine(lat1, lon1, lat2, lon2):
    R = 6371  # Radius of the Earth in kilometers
    dlat = radians(lat2 - lat1)
    dlon = radians(lon2 - lon1)
    a = sin(dlat / 2)**2 + cos(radians(lat1)) * cos(radians(lat2)) * sin(dlon / 2)**2
    c = 2 * atan2(sqrt(a), sqrt(1 - a))
    return R * c
    
# --- Helper Functions ---
def create_search_url(query, lang="en", geo_coordinates=None, zoom=12):
    """
    Creates a Google Maps search URL centered around specific geo coordinates.
    
    Args:
        query (str): Search query (e.g., "campus shoe store").
        lang (str): Language code (default: "en").
        geo_coordinates (tuple): (latitude, longitude) to center the map.
        zoom (int): Zoom level (default: 12).
    
    Returns:
        str: Full Google Maps search URL.
    """
    params = {'hl': lang}
    encoded_query = query.replace(' ', '+')
    if geo_coordinates:
        lat, lng = geo_coordinates
        return f"{BASE_URL}{encoded_query}/@{lat},{lng},{zoom}z?{urlencode(params)}"
    return f"{BASE_URL}{encoded_query}?{urlencode(params)}"

   

# --- Main Scraping Logic ---
async def scrape_google_maps(query, max_places=None, lang="en", headless=True, lat=None, lng=None, max_distance_km=30):
    results = []
    place_links = set()
    scroll_attempts_no_new = 0
   
    search_url = create_search_url(query, lang, geo_coordinates=(lat, lng))

    async with async_playwright() as p: # Changed to async
        try:
            browser = await p.chromium.launch(headless=headless) # Added await
            context = await browser.new_context( # Added await
                user_agent='Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36',
                java_script_enabled=True,
                accept_downloads=False,
                # Consider setting viewport, locale, timezone if needed
                locale=lang,
            )
           
            page = await context.new_page() # Added await
            if not page:
                await browser.close() # Close browser before raising
                if lat is not None and lng is not None:
                    results = [
                        place for place in results
                        if 'latitude' in place and 'longitude' in place and
                        haversine(lat, lng, place['latitude'], place['longitude']) <= max_distance_km
                    ]
                
                print(f"\nScraping finished. Found details for {len(results)} places.")
                return results
                raise Exception("Failed to create a new browser page (context.new_page() returned None).")
            # Removed problematic: await page.set_default_timeout(DEFAULT_TIMEOUT)
            # Removed associated debug prints

            search_url = create_search_url(query, lang, geo_coordinates=(lat, lng))
            print(f"Navigating to search URL: {search_url}")
            await page.goto(search_url, wait_until='domcontentloaded') # Added await
            await asyncio.sleep(2) # Changed to asyncio.sleep, added await

            # --- Handle potential consent forms ---
            # This is a common pattern, might need adjustment based on specific consent popups
            try:
                consent_button_xpath = "//button[.//span[contains(text(), 'Accept all') or contains(text(), 'Reject all')]]"
                # Wait briefly for the button to potentially appear
                await page.wait_for_selector(consent_button_xpath, state='visible', timeout=5000) # Added await
                # Click the "Accept all" or equivalent button if found
                # Example: Prioritize "Accept all"
                accept_button = await page.query_selector("//button[.//span[contains(text(), 'Accept all')]]") # Added await
                if accept_button:
                    print("Accepting consent form...")
                    await accept_button.click() # Added await
                else:
                    # Fallback to clicking the first consent button found (might be reject)
                    print("Clicking first available consent button...")
                    await page.locator(consent_button_xpath).first.click() # Added await
                # Wait for navigation/popup closure
                await page.wait_for_load_state('networkidle', timeout=5000) # Added await
            except PlaywrightTimeoutError:
                print("No consent form detected or timed out waiting.")
            except Exception as e:
                print(f"Error handling consent form: {e}")


            # --- Scrolling and Link Extraction ---
            print("Scrolling to load places...")
            feed_selector = '[role="feed"]'
            try:
                await page.wait_for_selector(feed_selector, state='visible', timeout=25000) # Added await
            except PlaywrightTimeoutError:
                 # Check if it's a single result page (maps/place/)
                if "/maps/place/" in page.url:
                    print("Detected single place page.")
                    place_links.add(page.url)
                else:
                    print(f"Error: Feed element '{feed_selector}' not found. Maybe no results? Taking screenshot.")
                    await page.screenshot(path='feed_not_found_screenshot.png') # Added await
                    await browser.close() # Added await
                    if lat is not None and lng is not None:
                        results = [
                            place for place in results
                            if 'latitude' in place and 'longitude' in place and
                            haversine(lat, lng, place['latitude'], place['longitude']) <= max_distance_km
                        ]
                    
                    print(f"\nScraping finished. Found details for {len(results)} places.")
                    return results
                    
                    return [] # No results or page structure changed

            if await page.locator(feed_selector).count() > 0: # Added await
                last_height = await page.evaluate(f'document.querySelector(\'{feed_selector}\').scrollHeight') # Added await
                while True:
                    # Scroll down
                    await page.evaluate(f'document.querySelector(\'{feed_selector}\').scrollTop = document.querySelector(\'{feed_selector}\').scrollHeight') # Added await
                    await asyncio.sleep(SCROLL_PAUSE_TIME) # Changed to asyncio.sleep, added await

                    # Extract links after scroll
                    current_links_list = await page.locator(f'{feed_selector} a[href*="/maps/place/"]').evaluate_all('elements => elements.map(a => a.href)') # Added await
                    current_links = set(current_links_list)
                    new_links_found = len(current_links - place_links) > 0
                    place_links.update(current_links)
                    print(f"Found {len(place_links)} unique place links so far...")

                    if max_places is not None and len(place_links) >= max_places:
                        print(f"Reached max_places limit ({max_places}).")
                        place_links = set(list(place_links)[:max_places]) # Trim excess links
                        break

                    # Check if scroll height has changed
                    new_height = await page.evaluate(f'document.querySelector(\'{feed_selector}\').scrollHeight') # Added await
                    if new_height == last_height:
                        # Check for the "end of results" marker
                        end_marker_xpath = "//span[contains(text(), \"You've reached the end of the list.\")]"
                        if await page.locator(end_marker_xpath).count() > 0: # Added await
                            print("Reached the end of the results list.")
                            break
                        else:
                            # If height didn't change but end marker isn't there, maybe loading issue?
                            # Increment no-new-links counter
                            if not new_links_found:
                                scroll_attempts_no_new += 1
                                print(f"Scroll height unchanged and no new links. Attempt {scroll_attempts_no_new}/{MAX_SCROLL_ATTEMPTS_WITHOUT_NEW_LINKS}")
                                if scroll_attempts_no_new >= MAX_SCROLL_ATTEMPTS_WITHOUT_NEW_LINKS:
                                    print("Stopping scroll due to lack of new links.")
                                    break
                            else:
                                scroll_attempts_no_new = 0 # Reset if new links were found this cycle
                    else:
                        last_height = new_height
                        scroll_attempts_no_new = 0 # Reset if scroll height changed

                    # Optional: Add a hard limit on scrolls to prevent infinite loops
                    # if scroll_count > MAX_SCROLLS: break

            # --- Scraping Individual Places ---
            print(f"\nScraping details for {len(place_links)} places...")
            count = 0
            for link in place_links:
                count += 1
                print(f"Processing link {count}/{len(place_links)}: {link}") # Keep sync print
                try:
                    await page.goto(link, wait_until='domcontentloaded') # Added await
                    # Wait a bit for dynamic content if needed, or wait for a specific element
                    # await page.wait_for_load_state('networkidle', timeout=10000) # Or networkidle if needed

                    html_content = await page.content() # Added await
                    place_data = extractor.extract_place_data(html_content)

                    if place_data:
                        place_data['link'] = link # Add the source link
                        results.append(place_data)
                        # print(json.dumps(place_data, indent=2)) # Optional: print data as it's scraped
                    else:
                        print(f"  - Failed to extract data for: {link}")
                        # Optionally save the HTML for debugging
                        # with open(f"error_page_{count}.html", "w", encoding="utf-8") as f:
                        #     f.write(html_content)

                except PlaywrightTimeoutError:
                    print(f"  - Timeout navigating to or processing: {link}")
                except Exception as e:
                    print(f"  - Error processing {link}: {e}")
                await asyncio.sleep(0.5) # Changed to asyncio.sleep, added await

            await browser.close() # Added await
            if lat is not None and lng is not None:
                results = [
                    place for place in results
                    if 'latitude' in place and 'longitude' in place and
                    haversine(lat, lng, place['latitude'], place['longitude']) <= max_distance_km
                ]
            
            print(f"\nScraping finished. Found details for {len(results)} places.")
            return results

        except PlaywrightTimeoutError:
            print(f"Timeout error during scraping process.")
        except Exception as e:
            print(f"An error occurred during scraping: {e}")
            import traceback
            traceback.print_exc() # Print detailed traceback for debugging
        finally:
            # Ensure browser is closed if an error occurred mid-process
            if 'browser' in locals() and browser.is_connected(): # Check if browser exists and is connected
                await browser.close() # Added await
                if lat is not None and lng is not None:
                    results = [
                        place for place in results
                        if 'latitude' in place and 'longitude' in place and
                        haversine(lat, lng, place['latitude'], place['longitude']) <= max_distance_km
                    ]
                
                print(f"\nScraping finished. Found details for {len(results)} places.")
                return results

# --- Example Usage ---
# (Example usage block removed as this script is now intended to be imported as a module)