/**
 * @file
 * A Backbone View that provides a dynamic contextual link.
 */

(function($, Backbone, Drupal) {
  Drupal.quickedit.ContextualLinkView = Backbone.View.extend(
    /** @lends Drupal.quickedit.ContextualLinkView# */ {
      /**
       * Define all events to listen to.
       *
       * @return {object}
       *   A map of events.
       */
      events() {
        // Prevents delay and simulated mouse events.
        function touchEndToClick(event) {
          event.preventDefault();
          event.target.click();
        }

        return {
          'click a': function(event) {
            event.preventDefault();
            this.model.set('state', 'launching');
          },
          'touchEnd a': touchEndToClick,
        };
      },

      /**
       * Create a new contextual link view.
       *
       * @constructs
       *
       * @augments Backbone.View
       *
       * @param {object} options
       *   An object with the following keys:
       * @param {Drupal.quickedit.EntityModel} options.model
       *   The associated entity's model.
       * @param {Drupal.quickedit.AppModel} options.appModel
       *   The application state model.
       * @param {object} options.strings
       *   The strings for the "Quick edit" link.
       */
      initialize(options) {
        // Insert the text of the quick edit toggle.
        this.$el.find('a').text(options.strings.quickEdit);
        // Initial render.
        this.render();
        // Re-render whenever this entity's isActive attribute changes.
        this.listenTo(this.model, 'change:isActive', this.render);
      },

      /**
       * Render function for the contextual link view.
       *
       * @param {Drupal.quickedit.EntityModel} entityModel
       *   The associated `EntityModel`.
       * @param {bool} isActive
       *   Whether the in-place editor is active or not.
       *
       * @return {Drupal.quickedit.ContextualLinkView}
       *   The `ContextualLinkView` in question.
       */
      render(entityModel, isActive) {
        this.$el.find('a').attr('aria-pressed', isActive);

        // Hides the contextual links if an in-place editor is active.
        this.$el.closest('.contextual').toggle(!isActive);

        return this;
      },
    },
  );
})(jQuery, Backbone, Drupal);
