<?php

namespace Drupal\custom_sitemap\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\Response;
use Drupal\node\Entity\Node;
use Drupal\taxonomy\Entity\Term;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for generating the custom XML sitemap.
 */
class SitemapController extends ControllerBase {

  /**
   * Generates the XML Sitemap.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The response containing the XML sitemap.
   */
  public function generateSitemap() {
    // Fetch URLs for nodes and taxonomy terms.
    $urls = $this->getUrls();

    // Build the XML sitemap structure.
    $sitemap = $this->buildSitemap($urls);

    // Create the response with the XML content type.
    $response = new Response($sitemap);
    $response->headers->set('Content-Type', 'application/xml');

    return $response;
  }

  /**
   * Fetch URLs for nodes and taxonomy terms.
   *
   * @return array
   *   An array of URLs and their metadata for the sitemap.
   */
  private function getUrls() {
    $urls = [];
    $urls[] = [
        'loc' => 'https://www.dic.co.in',
        'lastmod' => date('Y-m-d'),
        'changefreq' => 'daily',
        'priority' => '1.0',
      ];
    // Query nodes and add their URLs.
    $node_query = \Drupal::entityQuery('node')
      ->condition('type', ['page', 'products', 'article', 'news', 'print_problem_solution'], 'IN')
      ->condition('status', 1) // Only published nodes.
      ->sort('nid', 'ASC')
      ->execute();
    $nodes = Node::loadMultiple($node_query);

    foreach ($nodes as $node) {
      $urls[] = [
        'loc' => $node->toUrl('canonical', ['absolute' => TRUE])->toString(),
        'lastmod' => $node->getChangedTime() ? date('Y-m-d', $node->getChangedTime()) : date('Y-m-d'),
        'changefreq' => 'daily',
        'priority' => '0.9',
      ];
    }

    // Query taxonomy terms and add their URLs.
    $vocabularies = \Drupal::entityTypeManager()->getStorage('taxonomy_vocabulary')->loadMultiple();
    
    $vocabularies = array('investor', 'categories', 'print_problem_solution');
    foreach ($vocabularies as $vocabulary) {
      //dump($vocabulary->id());
      $term_query = \Drupal::entityQuery('taxonomy_term')
        ->condition('vid', $vocabulary)
        ->execute();
      $terms = Term::loadMultiple($term_query);

      foreach ($terms as $term) {
        $urls[] = [
          'loc' => $term->toUrl('canonical', ['absolute' => TRUE])->toString(),
          'lastmod' => $term->getChangedTime() ? date('Y-m-d', $term->getChangedTime()) : date('Y-m-d'),
          'changefreq' => 'monthly',
          'priority' => '0.8',
        ];
      }
    }

    return $urls;
  }

  /**
   * Builds the XML sitemap string from the array of URLs.
   *
   * @param array $urls
   *   An array of URL data.
   *
   * @return string
   *   The XML formatted sitemap.
   */
  private function buildSitemap(array $urls) {
    // Start the XML document.
    $sitemap = '<?xml version="1.0" encoding="UTF-8"?>';
    $sitemap .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">';

    // Add each URL entry in the XML structure.
    foreach ($urls as $url) {
      $sitemap .= '<url>';
      $sitemap .= '<loc>' . htmlspecialchars($url['loc']) . '</loc>';
      $sitemap .= '<lastmod>' . $url['lastmod'] . '</lastmod>';
      $sitemap .= '<changefreq>' . $url['changefreq'] . '</changefreq>';
      $sitemap .= '<priority>' . $url['priority'] . '</priority>';
      $sitemap .= '</url>';
    }

    $sitemap .= '</urlset>';

    return $sitemap;
  }

}
